#####################################################################
# Copyright 2002 Stefan Goetz
#####################################################################
# This file is part of vdrpylib.
# 
# vdrpylib is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# vdrpylib is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with vdrpylib; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#####################################################################

import string

class Channel:
	"""This class represents a program channel.
	
	It is intended to a container for the technical channel data as
	as for EPG Event objects.
	
	VDR may place a single physical channel at several locations in
	its global list of channels. This class does not represent a slot
	in the global list of channels but a physical channel. Physical
	channels are assumed to be uniquely identifiable by their SID.
	
	The indexes variable is a list containing the indexes of this
	channel in the global list of channels of a VDR instance.
	
	The name variable is a string containing the name of this
	channel.
	
	The freq variable is an integer containing the frequency of this
	channel in Hz.
	
	The pol variable is a string of length one containing the
	polariazation of this channel. 'h' stands for horizontal, 'v' for
	vertical polarization.
	
	The diseqc variable is an integer containing the DiSEqC code for
	this channel.
	
	The srate variable is an integer containing the symbol rate of
	this channel.
	
	The vpid variable is an integer containing the video PID of this
	channel.
	
	The apids variable is a list of integers containing the audio
	PIDs of this channel.
	
	The dpis variable is a list of integers containing the dolby
	digital (AC3) PIDs of this channel.
	
	The tpid variable is an integer containing the teletext PID of
	this channel.
	
	The ca variable is an integer defining how the channel can be
	accessed over the available DVB cards.
	
	The sid variable is an integer containing the service ID of this
	channel.
	
	The events variable is a list of Event objects sorted by start
	time. The list should not be manipulated directly but only
	through the XXXevent() functions of this class.
	"""
	def __init__(self, line = None, index = None):
		self.indexes = []
		self.name = None
		self.freq = None
		self.pol = None
		self.diseqc = None
		self.srate = None
		self.vpid = None
		self.apids = []
		self.dpids = []
		self.tpid = None
		self.ca = None
		self.sid = None
		self.events = []
		if line is not None:
			self.parse(line)
		if index is not None:
			self.indexes.append(index)
		
	def __str__(self):
		"""Creates a string representation of this channel object
		in the format of VDR's channels.conf file.
		"""
		apids = string.join([string.join(map(str, self.apids), ','), string.join(map(str, self.dpids), ',')], ';')
		return string.join(map(str, [self.name, self.freq, self.pol, self.diseqc, self.srate, self.vpid, apids, self.tpid, self.ca, self.sid]), ':')

	
	def parse(self, string):
		"""Parses a standard channel specification as can be found in
		VDR's channels.conf file.
		
		All fields of this channel object are set according to the 
		specification string given in the first argument.
		"""
		tokens = string.strip().split(':')
		self.name = tokens[0]
		self.freq = int(tokens[1])
		self.pol = tokens[2]
		self.diseqc = int(tokens[3])
		self.srate = int(tokens[4])
		self.vpid = int(tokens[5])
		apids = tokens[6]
		self.tpid = int(tokens[7])
		self.ca = int(tokens[8])
		self.sid = int(tokens[9])
		tokens = apids.split(';')
		self.apids = map(int, tokens[0].split(','))
		if len(tokens) == 2:
			self.dpids = map(int, tokens[1].split(','))
		
	def addevent(self, ev):
		"""Add an EPG event to the list of events for this channel.
		
		If the event has no ID one is assigned.
		
		The first argument shall be an Event object representing
		the event to be added. The start time of the event to add
		must be later than the end of any other event of this
		channel.
		
		The return value is the ID of the event if it satisfied the
		given restrictions and was added, else None is returned.
		"""
		if len(self.events) == 0 or self.events[-1].start + self.events[-1].dur <= ev.start:
			self.events.append(ev)
			if ev.id is None:
				if len(self.events) > 1:
					ev.id = (self.events[-2].id + 1) % 65536
				else:
					ev.id = 0
			return ev.id
		else:
			return None
	
	
	def getepgstr(self):
		"""Creates a string representation of this channel object
		in the format of VDR's epgdata file including all contained
		EPG events.
		"""
		s = 'C ' + str(self.sid)
		if self.name is not None:
			s = s + ' ' + self.name
		s = s + '\n'
		for ev in self.events:
			s = s + str(ev)
		s = s + 'c\n'
		return s
